import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { CheckCircle, Calendar, Clock, MapPin, Phone, Mail } from 'lucide-react'
import './BookTraining.css'

const BookTraining = () => {
  const navigate = useNavigate()
  const [selectedType, setSelectedType] = useState(null)
  const [selectedDate, setSelectedDate] = useState(null)
  const [selectedTime, setSelectedTime] = useState(null)

  const trainingTypes = [
    { id: 'yoga', name: 'Йога', icon: '🧘', color: '#8b5cf6' },
    { id: 'crossfit', name: 'Кроссфит', icon: '💪', color: '#ef4444' },
    { id: 'pilates', name: 'Пилатес', icon: '🤸', color: '#10b981' },
    { id: 'stretching', name: 'Стретчинг', icon: '🧘‍♀️', color: '#f59e0b' }
  ]

  const availableDates = [
    '2025-03-20',
    '2025-03-21',
    '2025-03-22',
    '2025-03-23',
    '2025-03-24',
    '2025-03-25',
    '2025-03-26'
  ]

  const availableTimes = [
    '09:00', '10:00', '11:00', '12:00',
    '17:00', '18:00', '19:00', '20:00'
  ]

  const trainers = {
    'yoga': 'Анна Петрова',
    'crossfit': 'Иван Сидоров',
    'pilates': 'Мария Козлова',
    'stretching': 'Ольга Новикова'
  }

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedType && selectedDate && selectedTime) {
      const savedBookings = localStorage.getItem('fitnessBookings')
      const bookings = savedBookings ? JSON.parse(savedBookings) : { active: [], past: [] }
      
      const newBooking = {
        id: Date.now(),
        type: trainingTypes.find(t => t.id === selectedType).name,
        date: selectedDate,
        time: selectedTime,
        trainer: trainers[selectedType] || 'Тренер',
        status: 'active'
      }

      bookings.active.push(newBooking)
      localStorage.setItem('fitnessBookings', JSON.stringify(bookings))
      
      setSelectedType(null)
      setSelectedDate(null)
      setSelectedTime(null)
      
      navigate('/')
    }
  }

  const benefits = [
    { icon: '💪', title: 'Профессиональные тренеры', desc: 'Опыт работы более 5 лет' },
    { icon: '🏋️', title: 'Современное оборудование', desc: 'Новейшие тренажеры и инвентарь' },
    { icon: '⏰', title: 'Гибкий график', desc: 'Тренировки в удобное время' }
  ]

  return (
    <div className="book-training-page">
      <div className="page-header">
        <h1 className="page-title">Запись на тренировку</h1>
        <p className="page-description">Выберите удобное время и тип тренировки</p>
      </div>

      <div className="benefits-section">
        <h2 className="benefits-title">Почему выбирают нас</h2>
        <div className="benefits-grid">
          {benefits.map((benefit, index) => (
            <div key={index} className="benefit-card">
              <span className="benefit-icon">{benefit.icon}</span>
              <h3 className="benefit-title">{benefit.title}</h3>
              <p className="benefit-desc">{benefit.desc}</p>
            </div>
          ))}
        </div>
      </div>

      <form onSubmit={handleSubmit} className="booking-form">
        <div className="form-section">
          <h2 className="section-title">
            <span className="title-icon">🏋️</span>
            Выберите вид тренировки
          </h2>
          <div className="training-types">
            {trainingTypes.map((type) => (
              <button
                key={type.id}
                type="button"
                className={`type-card ${selectedType === type.id ? 'selected' : ''}`}
                onClick={() => setSelectedType(type.id)}
                style={{
                  '--type-color': type.color
                }}
              >
                <span className="type-icon">{type.icon}</span>
                <span className="type-name">{type.name}</span>
                {selectedType === type.id && (
                  <CheckCircle className="check-icon" size={20} />
                )}
              </button>
            ))}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <Calendar className="section-icon" size={24} />
            Выберите дату
          </h2>
          <div className="dates-grid">
            {availableDates.map((date) => {
              const dateObj = new Date(date)
              const isSelected = selectedDate === date
              return (
                <button
                  key={date}
                  type="button"
                  className={`date-card ${isSelected ? 'selected' : ''}`}
                  onClick={() => setSelectedDate(date)}
                >
                  <div className="date-day">{dateObj.toLocaleDateString('ru-RU', { day: 'numeric' })}</div>
                  <div className="date-month">{dateObj.toLocaleDateString('ru-RU', { month: 'short' })}</div>
                </button>
              )
            })}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <Clock className="section-icon" size={24} />
            Выберите время
          </h2>
          <div className="times-grid">
            {availableTimes.map((time) => (
              <button
                key={time}
                type="button"
                className={`time-card ${selectedTime === time ? 'selected' : ''}`}
                onClick={() => setSelectedTime(time)}
              >
                {time}
              </button>
            ))}
          </div>
        </div>

        <button
          type="submit"
          className="submit-btn"
          disabled={!selectedType || !selectedDate || !selectedTime}
        >
          Записаться на тренировку
        </button>
      </form>

      <div className="contact-section">
        <h2 className="contact-title">Контакты</h2>
        <div className="contact-info">
          <div className="contact-item">
            <MapPin className="contact-icon" size={20} />
            <span>г. Москва, ул. Фитнесная, д. 10</span>
          </div>
          <div className="contact-item">
            <Phone className="contact-icon" size={20} />
            <span>+7 (495) 123-45-67</span>
          </div>
          <div className="contact-item">
            <Mail className="contact-icon" size={20} />
            <span>info@fitflow.ru</span>
          </div>
        </div>
      </div>
    </div>
  )
}

export default BookTraining

